<?php

namespace TsLib\Rest\Services;

use TsLib\Rest\Models\NsRestlet;
use TsLib\Rest\Models\NsRestletRequest;

use Illuminate\Support\Facades\Log;

class NetsuiteRest {

    protected $tokenKey;
    protected $tokenSecret;
    protected $consumerKey;
    protected $consumerSecret;
    protected $signatureMethod;
    protected $nonce;
    protected $timestamp;
    protected $version;
    protected $realm;

    public function __construct()
    {
        date_default_timezone_set(config('app.timezone', 'America/Mexico_City'));
        $this->tokenKey =  config('ts-lib-rest.token_key');
        $this->tokenSecret = config('ts-lib-rest.token_secret');
        $this->consumerKey = config('ts-lib-rest.consumer_key');
        $this->consumerSecret = config('ts-lib-rest.consumer_secret');
        $this->realm = config('ts-lib-rest.account');
        $this->signatureMethod = "HMAC-SHA256";
        $this->version = "1.0";
    }

    public function restlet($httpMethod, $name, $params, $json='')
    {
        $url = $this->make_restlet_url($name, $params);

        return $this->doRequest($httpMethod, $url, $json);
    }

    public function rest($httpMethod, $path, $json = '')
    {
        $url = sprintf('https://%s.suitetalk.api.netsuite.com/services/rest/', strtolower(str_replace('_', '-', $this->realm))).$path;

        return $this->doRequest($httpMethod, $url, $json);
    }

    private function doRequest($httpMethod,$url,$json="")
    {
        $resp = ["status" => false,"data" => [],"message"=>"Error api get"];

        $log_request = new NsRestletRequest();
        $log_request->url = $url;
        $log_request->method = $httpMethod;
        $log_request->json = $json;
        $log_request->status = 0;
        $log_request->save();

        if($httpMethod == "POST"){
            $validateJSON = $this->_jsonValidator($json);
            if($validateJSON != 1){
                $response["message"] = "JSON error";
                return $resp;
            }
        }

        $header = [
            $this->_getOauth($httpMethod, $url),
            'Content-type:application/json'
        ];

        if($httpMethod == "POST"){
            $options = [
                CURLOPT_URL => $url,  
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => "",
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => $httpMethod,
                CURLOPT_POSTFIELDS => $json,
                CURLOPT_HTTPHEADER => $header,
                CURLOPT_HEADER => 1
            ];
        }else{
            $options = [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => "",
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => $httpMethod,
                CURLOPT_HTTPHEADER => $header,
                CURLOPT_HEADER => true,
            ];
        }

        $curl = curl_init();
        curl_setopt_array($curl, $options);
        $response = curl_exec($curl);
        $headerSize = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
        $header = substr($response, 0, $headerSize);
        $bodyHead = substr($response, $headerSize);

        if($response === false)
        {
            $log_request->response = curl_error($curl);
            $log_request->status = 1;
            $log_request->save();
            return $resp;
        }

        $result = json_decode($response);
        $resp["message"] = json_decode($response);

        $arrayHead = explode("\\r\\n",json_encode($header));
        $responseHeaders = [];

        $responseCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        foreach ($arrayHead as $head) {
            $parts = explode(':', $head);
            if(isset($parts[1]))
            {
                $responseHeaders[trim($parts[0])] = trim($parts[1]);
            }
            else
            {
                $responseHeaders[] = trim($head);
            }
        }

        curl_close($curl);

        if($httpMethod == "POST"){
            $resultNS = json_decode($bodyHead);
           
            if($resultNS->code){
                $resp["data"] = $resultNS;
                $resp["status"] = true;
            }

            if(isset($responseHeaders['Location']))
            {
                $resp["data"] = $responseHeaders['Location'];
                $resp["status"] = true;
            }
        }

        if($httpMethod == "GET"){
            $result = json_decode($bodyHead);
            if($result){
                if(isset($result->code) && $result->code)
                {
                    $resp["message"] = $result->message;
                    $resp["data"] = $result->data;
                    $resp["status"] = true;
                }
                else
                {
                    $resp['data'] = $result;
                    $resp['status'] = false;
                    $log_request->status = 1;
                }
            }
        }

        $log_request->response = $bodyHead;
        if($resp["status"])
        {
            $log_request->status = 9;
        }
        $log_request->save();

        return $resp;
    }

    private function make_restlet_url($name, $params = [])
    {
        $restlet = NsRestlet::where('name', $name)->first();
        if($restlet == null)
        {
            Log::error(sprintf("Se intento llamar al Restlet %s pero no tiene registro en ns_restlets", $name));
            return false;
        }

        if(strpos($this->realm, '_') === false)
        {
            $script_id = $restlet->script_id;
        }
        else
        {
            $script_id = $restlet->script_id_sb;
        }

        if ($restlet->has_debug_deploy === 0)
        {
            $deploy = 1;
        }
        else
        {
            $deploy = $restlet->has_debug_deploy;
        }

        $query = array_merge([
                "script" => $script_id,
                "deploy" => $deploy,
            ], $params);

        return strtolower(sprintf("https://%s.restlets.api.netsuite.com/app/site/hosting/restlet.nl?%s", strtoupper(str_replace('_', '-', $this->realm)), http_build_query($query)));
    }

   public function _jsonValidator($json)
    {
        if (!empty($json)) {
            @json_decode($json);
            return (json_last_error() === JSON_ERROR_NONE);
        }
        return false;
    }

    public function _wrapperAcentos($text="")
    {
        return self::wrapperAcentos($text);
    }

    static public function wrapperAcentos($text="")
    {
        return htmlentities(html_entity_decode(utf8_encode($text)));
    }

    private function _getOauth($httpMethod, $url)
    {
        $nonce = substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 20);
        $timestamp = time();

        if (strpos($url, '?')){  
            $baseUrl = substr($url, 0, strpos($url, '?'));
            $getParams = substr($url, strpos($url, '?') + 1);
        } else {
            $baseUrl = $url;
            $getParams = "";
        }

        $restlet = [];
        if($getParams!=""){
            $restlet = explode('&',$getParams);
        }

        $baseString = strtoupper($httpMethod).'&'.rawurlencode($baseUrl).'&';

        $params = $values = [];
        foreach($restlet as $par){
            $val = explode("=",$par);
            $values[]=$val[0];
        }
       
        if(in_array("customer_id",$values)){
            $params["customer_id"] = substr($restlet[2],strpos($restlet[2], '=')+1,10);
        }
        if(in_array("deploy",$values)){
            $params["deploy"] = substr($restlet[1],strpos($restlet[1], '=')+1,10);
        }
        if(in_array("month",$values)){
            $params["month"] = substr($restlet[3],strpos($restlet[3], '=')+1,10);
        }
        $params['oauth_consumer_key'] = $this->consumerKey;
        $params['oauth_nonce'] = $nonce;
        $params['oauth_signature_method'] = $this->signatureMethod;
        $params['oauth_timestamp'] = $timestamp;
        $params['oauth_token'] = $this->tokenKey;
        $params['oauth_version'] = $this->version;
        if(in_array("script",$values)){
            $params["script"] = substr($restlet[0],strpos($restlet[0], '=')+1,10);
        }
        if(in_array("tranid",$values)){
            $params["tranid"] = substr($restlet[2],strpos($restlet[2], '=')+1,10);
        }
        if(in_array("year",$values)){
            $params["year"] = substr($restlet[4],strpos($restlet[4], '=')+1,10);
        }
        
        $paramString = "";
        foreach ($params as $key => $value){
            $paramString .= rawurlencode($key) . '='. rawurlencode($value) .'&';
        }

        $paramString = substr($paramString, 0, -1);
        $baseString .= rawurlencode($paramString);

        $key = rawurlencode($this->consumerSecret) .'&'. rawurlencode($this->tokenSecret);

        $signature = base64_encode(hash_hmac('sha256', $baseString, $key, true));

        return 'Authorization: OAuth '
                        .'realm="' .rawurlencode($this->realm) .'", '
                        .'oauth_consumer_key="' .rawurlencode($this->consumerKey) .'", '
                        .'oauth_token="' .rawurlencode($this->tokenKey) .'", '
                        .'oauth_nonce="' .rawurlencode($nonce) .'", '
                        .'oauth_timestamp="' .rawurlencode($timestamp) .'", '
                        .'oauth_signature_method="' .rawurlencode($this->signatureMethod) .'", '
                        .'oauth_version="' .rawurlencode($this->version) .'", '
                        .'oauth_signature="' .rawurlencode($signature) .'"';
    }
}